-- (c) Meta Platforms, Inc. and affiliates. Confidential and proprietary.

function table.contains(table, element)
    for _, value in pairs(table) do
        if value == element then
            return true
        end
    end
    return false
  end

  -- Get the median of a table.
  function table.median( t )
    local temp={}

    -- deep copy table so that when we sort it, the original is unchanged
    -- also weed out any non numbers
    for k,v in pairs(t) do
      if type(v) == 'number' then
        table.insert( temp, v )
      end
    end

    table.sort( temp )

    -- If we have an even number of table elements or odd.
    if math.fmod(#temp,2) == 0 then
      -- return mean value of middle two elements
      return ( temp[#temp/2] + temp[(#temp/2)+1] ) / 2
    else
      -- return middle element
      return temp[math.ceil(#temp/2)]
    end
  end


function getConfig()
    -- default config
    local config = {
        maxItemCount = 15,
        isVideoSegmentationEnabled = false,
        minVideoSegmentDuration = 30,
        seedValue = 0
    }

    for _, mediaEvent in ipairs(MediaEvents) do
        if mediaEvent:getSourceMedia() == "config" then
            for i = 0, mediaEvent:size() - 1, 1 do
                local startTime, weight, duration, tags = mediaEvent:getEvent(i)
                local param = tags[1]
                if param == "maxItemCount" then
                    config.maxItemCount = math.floor(weight)
                elseif param == "isVideoSegmentationEnabled" then
                    config.isVideoSegmentationEnabled = weight == 1.0
                elseif param == "minVideoSegmentDuration" then
                    config.minVideoSegmentDuration = math.floor(weight)
                elseif param == "seed" then
                    config.seedValue = math.floor(weight)
                end
            end
        end
    end

    return config
end

function getBeatInfo()
    local beatsTable = {}
    local prevStart = 0
    local beatDurations = {}
    local medianDuration = -1
    for _, mediaEvent in ipairs(MediaEvents) do
        if mediaEvent:getSourceMedia() == "config" then goto CONTINUE end
        local numEvents = mediaEvent:size()
        local onsetsAfter = 0
        for i = 0, numEvents - 1, 1 do
        -- check 'beat' tag in an event
        local start, weights, _, tags = mediaEvent:getEvent(i)
        if (table.contains(tags, "beat") and table.contains(tags, "onset") ) then
            -- save previous beat's onsets after the beat
            if #beatsTable ~= 0 then
                beatsTable[#beatsTable].onsetsAfter = onsetsAfter
            end
            -- save the new beat
            local beat = {
                startTime = start,
                weight = weights,
                onset = table.contains(tags, "onset"),
                onsetsBefore = onsetsAfter, -- onsets before beat are previous beat's onsets after the beat
                onsetsAfter = 0, -- This will be updated by the next beat.
            }
            table.insert(beatsTable, beat)
            onsetsAfter = 0
            -- save the beat duration
            local currDuration = start - prevStart
            table.insert(beatDurations, currDuration)
            prevStart = start
        else
            if #beatsTable ~= 0 then
                onsetsAfter = onsetsAfter + 1
            end
        end
        end
        ::CONTINUE::
    end

    if #beatDurations > 1 then
        medianDuration = table.median(beatDurations)
    end

    return beatsTable, medianDuration
end

PAN_RIGHT = [[{ "pan_style": 0 }]];
PAN_LEFT = [[{ "pan_style": 1 }]];
PAN_DOWN = [[{ "pan_style": 2 }]];
PAN_UP = [[{ "pan_style": 3 }]];

panStyles = {PAN_LEFT, PAN_UP, PAN_RIGHT, PAN_DOWN}

function main()
    -- Keep only beats that fit in the suggested timeline
    -- https://www.wolframalpha.com/input?i=0.02+*+%28x+%5E+2%29+-+0.6+*+x+%2B+6%2C+x+from+1.0+to+15&fbclid=IwAR28NqOXDY9gC2qRCoc5yoEzfyEdBjtZzgz4-XIP2Gw7rqxDu3r7oBg5etM
    local numMediaItems = #MediaItems
    local suggestedClipDuration = 0.02 * (numMediaItems ^ 2) - 0.6 * numMediaItems + 6
    local maxTimelineDuration = suggestedClipDuration * numMediaItems

    -- Find beatDuration
    local _, beatDuration = getBeatInfo()

    -- Create clips
    -- Number of clips must be equal to the number of imported media assets
    local transitionDur = 0.4
    local panIdx = 1
    for i = 1, numMediaItems, 1 do
        -- Sync clip to the beat
        local clipDuration = suggestedClipDuration
        if (beatDuration > 0.0 and suggestedClipDuration > beatDuration) then
            local beatsCount = math.floor(suggestedClipDuration/beatDuration)
            clipDuration = beatsCount * beatDuration
        end

        local mediaItem = MediaItems[i]
        local mediaItemStartTime = mediaItem:getStartSec()
        if (mediaItem:isVideo()) then
            local mediaDuration = mediaItem:getDurationSec()
            if (mediaDuration < clipDuration) then
                -- In this case ignore the beat
                clipDuration = mediaDuration
            end
        end

        Project:addClip(mediaItemStartTime, clipDuration, 1.0, mediaItem)
        Project:addTransition("whipPan", transitionDur, panStyles[panIdx]);

        panIdx = panIdx + 1
        if panIdx > #panStyles then
            panIdx = 1
        end
    end
end

main()
