-- (c) Meta Platforms, Inc. and affiliates. Confidential and proprietary.

-- Get the median of a table.
function table.median( t )
    local temp={}

    -- deep copy table so that when we sort it, the original is unchanged
    -- also weed out any non numbers
    for k,v in pairs(t) do
      if type(v) == 'number' then
        table.insert( temp, v )
      end
    end

    table.sort( temp )

    -- If we have an even number of table elements or odd.
    if math.fmod(#temp,2) == 0 then
      -- return mean value of middle two elements
      return ( temp[#temp/2] + temp[(#temp/2)+1] ) / 2
    else
      -- return middle element
      return temp[math.ceil(#temp/2)]
    end
end

function table.contains(table, element)
    for _, value in pairs(table) do
        if value == element then
            return true
        end
    end
    return false
end

function getConfig()
    -- default config
    local config = {
        maxItemCount = 15,
        isVideoSegmentationEnabled = false,
        minVideoSegmentDuration = 30,
        seedValue = 0,
        shouldAllowDuplicatedMedia = true,
        shouldEnableSingleMediaLogic = false
    }

    for _, mediaEvent in ipairs(MediaEvents) do
        if mediaEvent:getSourceMedia() == "config" then
            for i = 0, mediaEvent:size() - 1, 1 do
                local startTime, weight, duration, tags = mediaEvent:getEvent(i)
                local param = tags[1]
                if param == "maxItemCount" then
                    config.maxItemCount = math.floor(weight)
                elseif param == "isVideoSegmentationEnabled" then
                    config.isVideoSegmentationEnabled = weight == 1.0
                elseif param == "minVideoSegmentDuration" then
                    config.minVideoSegmentDuration = math.floor(weight)
                elseif param == "seed" then
                    config.seedValue = math.floor(weight)
                elseif param == "shouldEnableSingleMediaLogic" then
                    config.shouldEnableSingleMediaLogic = weight == 1.0
              end
            end
        end
    end

    return config
end

function isMediaEventsEmpty()
    local hasConfig = false
    for _, mediaEvent in ipairs(MediaEvents) do
        if mediaEvent:getSourceMedia() == "config" then
        hasConfig = true
        end
    end

    if hasConfig then
        return #MediaEvents == 1
    else
        return #MediaEvents == 0
    end
end

-- Extract only the beats and other info about them
function getBeatInfo()
    local beatsTable = {}
    local prevStart = 0
    local beatDurations = {}
    local medianDuration = -1
    for _, mediaEvent in ipairs(MediaEvents) do
        if mediaEvent:getSourceMedia() == "config" then goto CONTINUE end
        local numEvents = mediaEvent:size()
        local onsetsAfter = 0
        for i = 0, numEvents - 1, 1 do
        -- check 'beat' tag in an event
        local start, weights, _, tags = mediaEvent:getEvent(i)
        if (table.contains(tags, "beat") and table.contains(tags, "onset") ) then
            -- save previous beat's onsets after the beat
            if #beatsTable ~= 0 then
                beatsTable[#beatsTable].onsetsAfter = onsetsAfter
            end
            -- save the new beat
            local beat = {
                startTime = start,
                weight = weights,
                onset = table.contains(tags, "onset"),
                onsetsBefore = onsetsAfter, -- onsets before beat are previous beat's onsets after the beat
                onsetsAfter = 0, -- This will be updated by the next beat.
            }
            table.insert(beatsTable, beat)
            onsetsAfter = 0
            -- save the beat duration
            local currDuration = start - prevStart
            table.insert(beatDurations, currDuration)
            prevStart = start
        else
            if #beatsTable ~= 0 then
                onsetsAfter = onsetsAfter + 1
            end
        end
        end
        ::CONTINUE::
    end

    if #beatDurations > 1 then
        medianDuration = table.median(beatDurations)
    end

    return beatsTable, medianDuration
end

ZOOM_IN = [[
    {
      "start_amount": 0.0,
      "end_amount": 0.09,
      "curve": "LINEAR"
    }
]];

ZOOM_OUT = [[
    {
    "start_amount": 0.1,
    "end_amount": 0.0,
    "curve": "LINEAR"
    }
]];

GRADIENT_ROSE_ORANGE = [[{
    "color1_r": 1.0,
    "color1_g": 0.0,
    "color1_b": 1.0,
    "color2_r": 1.0,
    "color2_g": 0.4,
    "color2_b": 0.0
}]]

GRADIENT_ORANGE_WHITE = [[{
    "alpha": 0.3,
    "color1_r": 1.0,
    "color1_g": 0.4,
    "color1_b": 0.0,
    "color2_r": 1.0,
    "color2_g": 1.0,
    "color2_b": 1.0
}]]

BRIGHTNESS_1 = [[{
    "end_amount": 0.6,
    "curve": "LINEAR"
}]]

BRIGHTNESS_2 = [[{
    "start_amount": 0.6,
    "end_amount": 0.0,
    "curve": "LINEAR"
}]]

BRIGHTNESS_3 = [[{
    "end_amount": 1.0,
    "curve": "LINEAR"
}]]

FINAL_TRANSFORM = [[
    {
    "start_zoom": 1.1,
    "end_zoom": 1.25,
    "start_curve": "LINEAR",
    "end_curve": "EASE_OUT_SINE"
    }
]]

SINGLE_TRANSFORM = [[
    {
    "start_zoom": 1.0,
    "mid_zoom": 1.30,
    "end_zoom": 1.1,
    "start_curve": "LINEAR",
    "end_curve": "LINEAR"
    }
]]

function addClip(clipDuration, mediaIndex)
    local mediaItem = MediaItems[mediaIndex]
    local startTime = mediaItem:getStartSec()
    local speed = 1.0
    Project:addClip(startTime, clipDuration, speed, mediaItem)
end

function addSingleMediaEffect(clipDuration)
    -- SINGLE MEDIA FLOW
    Treatments:transformTreatment(0.0, 3*clipDuration/4, SINGLE_TRANSFORM)
    Treatments:brightnessTreatment(clipDuration/4, 1.5*clipDuration/4, BRIGHTNESS_1)
    Treatments:brightnessTreatment(1.5*clipDuration/4, 2*clipDuration/4, BRIGHTNESS_2)
    Treatments:transformTreatment(3*clipDuration/4, clipDuration, FINAL_TRANSFORM)
    Project:addEffect("fadeOut", clipDuration / 4)
end

function addEffect(clipIndex, totalNumClips, clipDuration)
    if (clipIndex % 2 == 0) then
        Treatments:zoomTreatment2(0.0, clipDuration, ZOOM_OUT)
    else
        Treatments:zoomTreatment2(0.0, clipDuration, ZOOM_IN)
    end

    local middleClipIndex = 7
    if (math.fmod(clipIndex, middleClipIndex) == 3) then
        Treatments:brightnessTreatment(clipDuration - clipDuration/3, clipDuration, BRIGHTNESS_1)
    end

    if (math.fmod(clipIndex, middleClipIndex) == 4) then
        Treatments:brightnessTreatment(0.0, clipDuration/3, BRIGHTNESS_2)
    end

    if (math.fmod(clipIndex, middleClipIndex) == 6) then
        Treatments:brightnessTreatment(clipDuration/2, clipDuration, BRIGHTNESS_3)
    end

    if (math.fmod(clipIndex, middleClipIndex) == 7) then
        Treatments:linearGradientTreatment(0.0, clipDuration, GRADIENT_ORANGE_WHITE);
    end

    if (clipIndex == totalNumClips) then
        Treatments:transformTreatment(0.0, clipDuration, FINAL_TRANSFORM)
        Project:addEffect("fadeOut", clipDuration / 2)
    end
end

function main()
    -- Keep only beats that fit in the suggested timeline
    -- https://www.wolframalpha.com/input?i=0.02+*+%28x+%5E+2%29+-+0.6+*+x+%2B+6%2C+x+from+1.0+to+15&fbclid=IwAR28NqOXDY9gC2qRCoc5yoEzfyEdBjtZzgz4-XIP2Gw7rqxDu3r7oBg5etM
    local numMediaItems = #MediaItems
    local config = getConfig()
    local suggestedClipDuration = 0.02 * (numMediaItems ^ 2) - 0.6 * numMediaItems + 6

    local singleMedia = (numMediaItems == 1 and config.shouldEnableSingleMediaLogic)
    if (singleMedia) then
        suggestedClipDuration = 8.0
    end

    -- Find beatDuration
    local _, beatDuration = getBeatInfo()

    -- if beats fit in suggested clip duration, follow the beat
    if (beatDuration > 0.0 and suggestedClipDuration > beatDuration) then
        local beatsCount = math.floor(suggestedClipDuration/beatDuration)
        suggestedClipDuration = beatsCount * beatDuration
    end

    for i = 1, numMediaItems, 1 do
        local clipDuration = suggestedClipDuration
        local mediaItem = MediaItems[i]
        if (mediaItem:isVideo()) then
            local mediaDuration = mediaItem:getDurationSec()
            if (mediaDuration < clipDuration) then
                -- In this case ignore the beat
                clipDuration = mediaDuration
            end
        end

        addClip(clipDuration, i)
        if (singleMedia) then
            addSingleMediaEffect(clipDuration)
        else
            addEffect(i, numMediaItems, clipDuration)
        end
    end
end

main()
