-- (c) Meta Platforms, Inc. and affiliates. Confidential and proprietary.

-- effect types
PAN_UP = [[
    {
    "zoom_scale": 0.1,
    "starting_x": 0.5,
    "starting_y": 0.45,
    "ending_x": 0.5,
    "ending_y": 0.55,
    "curve": "LINEAR"
    }
]]
PAN_DOWN = [[
    {
    "zoom_scale": 0.1,
    "starting_x": 0.5,
    "starting_y": 0.55,
    "ending_x": 0.5,
    "ending_y": 0.45,
    "curve": "LINEAR"
    }
]]
PAN_RIGHT = [[
    {
    "zoom_scale": 0.1,
    "starting_x": 0.45,
    "starting_y": 0.5,
    "ending_x": 0.55,
    "ending_y": 0.5,
    "curve": "LINEAR"
    }
]]
PAN_LEFT = [[
    {
    "zoom_scale": 0.1,
    "starting_x": 0.55,
    "starting_y": 0.5,
    "ending_x": 0.45,
    "ending_y": 0.5,
    "curve": "LINEAR"
    }
]]
ZOOM_IN = [[
    {
      "start_amount": 0.0,
      "end_amount": 0.1,
      "curve": "LINEAR"
    }
  ]];
ZOOM_OUT = [[
    {
      "start_amount": 0.1,
      "end_amount": 0.0,
      "curve": "LINEAR"
    }
  ]];
SEPIA = [[{
    "darken": 0.1,
    "sepia_amount": 0.1
}]]

effects = {
    PAN_UP,
    PAN_DOWN,
    PAN_RIGHT,
    PAN_LEFT,
    ZOOM_IN,
    ZOOM_IN,
    ZOOM_OUT,
    ZOOM_OUT,
}

-- util functions
function table.contains(table, element)
    for _, value in pairs(table) do
        if value == element then
            return true
        end
    end
    return false
end

function table.size(table)
    local ret = 0
    for _, v in pairs(table) do
        ret = ret + 1
    end
    return ret
end

function sortFunc(a, b)
    if a.onset and not b.onset then
        return true
    elseif b.onset and not a.onset then
        return false
    elseif a.weights > b.weights then
        return true
    elseif a.weights < b.weights then
        return false
    elseif a.diffToSuggestion < b.diffToSuggestion then
        return true
    elseif a.diffToSuggestion > b.diffToSuggestion then
        return false
    else
        return false
    end
end

function pickBestBeat(inputCandidates)
    table.sort(inputCandidates, sortFunc)

    for k, v in pairs(inputCandidates) do
        if (k == 1) then
            return v
        end
    end
end

function generateProjectWithEmptyBeatEvents(duration)
    for i = 1, table.size(_mediaItemsWithSegments), 1 do
        local mediaItem = MediaItems[_mediaItemsWithSegments[i].originalIndex]
        Project:addClip(0.0, duration, 1.0, mediaItem)
        if (mediaItem:isImage()) then
            addTreatment()
        end

        Treatments:sepiaTreatment(0.0, duration, SEPIA)

        if i < table.size(_mediaItemsWithSegments) then
            Project:addTransition("fade", _transitionDuration)
        end
    end
    Project:addEffect("fadeOut", _fadeOutDuration)
end

function addTreatment()
    local treatmentStart = 0.0
    local treatmentEnd = 6.0
    local effect = effects[math.random(#(effects))]
    if effect == PAN_UP or effect == PAN_DOWN or effect == PAN_RIGHT or effect == PAN_LEFT then
        Treatments:panTreatment(treatmentStart, treatmentEnd, effect);
    elseif effect == ZOOM_IN or effect == ZOOM_OUT then
        Treatments:zoomTreatment2(treatmentStart, treatmentEnd, effect);
    end
end

function shouldAddFadeTransition(mediaIndex, mediaItemsSize)
    return mediaIndex < mediaItemsSize
end


function generateMediaItemsWithSegments()
    local mediaItemsCount = table.size(MediaItems)
    local availableSegmentCount = _config.maxItemCount - mediaItemsCount
    local list = {}
    local offset = 0
    for i = 1, mediaItemsCount, 1 do
        local item = MediaItems[i]
        local startTime = item:getStartSec()
        local duration = item:getDurationSec()

        list[i + offset] = {}
        list[i + offset]["originalIndex"] = i
        list[i + offset]["startTime"] =  startTime

        if _config.isVideoSegmentationEnabled and not item:isImage() and duration > _config.minVideoSegmentDuration and availableSegmentCount > 0 then
            local interval = duration / 3
            for j = 1, 2, 1 do
                if availableSegmentCount > 0 then
                    offset = offset + 1
                    availableSegmentCount = availableSegmentCount - 1

                    list[i + offset] = {}
                    list[i + offset]["originalIndex"] = i
                    list[i + offset]["startTime"] = startTime + (interval * j)
                end
            end
        end
    end
    return list
end

function getConfig()
    -- default config
    local config = {
        maxItemCount = 15,
        isVideoSegmentationEnabled = false,
        minVideoSegmentDuration = 30,
        seedValue = 0
    }

    for _, mediaEvent in ipairs(MediaEvents) do
        if mediaEvent:getSourceMedia() == "config" then
            for i = 0, mediaEvent:size() - 1, 1 do
                local startTime, weight, duration, tags = mediaEvent:getEvent(i)
                local param = tags[1]
                if param == "maxItemCount" then
                    config.maxItemCount = math.floor(weight)
                elseif param == "isVideoSegmentationEnabled" then
                    config.isVideoSegmentationEnabled = weight == 1.0
                elseif param == "minVideoSegmentDuration" then
                    config.minVideoSegmentDuration = math.floor(weight)
                elseif param == "seed" then
                    config.seedValue = math.floor(weight)
                end
            end
        end
    end

    return config
end

function isMediaEventsEmpty()
    local configFilesAmt = 0
    for _, mediaEvent in ipairs(MediaEvents) do
        if mediaEvent:getSourceMedia() == "config" then
            configFilesAmt = configFilesAmt + 1
        end
    end

    local resultingTableSize = table.size(MediaEvents) - configFilesAmt
    return resultingTableSize == 0

end


-- const values
_trimThreshold = 1.0
_fadeOutDuration = 0.5
_config = getConfig()
_mediaItemsWithSegments = generateMediaItemsWithSegments()
_transitionDuration = 0.6

-- main function
function main()
    -- set seed if necessary
    if _config.seedValue > 0 then
        math.randomseed(_config.seedValue)
    end

    -- media items size
    local mediaItemsSize = table.size(_mediaItemsWithSegments)

    -- calculate minimum clip duration given the media item size
    local suggestedDuration = 0.02 * (mediaItemsSize ^ 2) - 0.6 * mediaItemsSize + 6

    -- if no media event exists, make transitions at suggested duration
    if (isMediaEventsEmpty()) then
        generateProjectWithEmptyBeatEvents(suggestedDuration)
        return
    end

    -- initial item info
    local mediaItemIndex = 1
    local mediaItem = _mediaItemsWithSegments[mediaItemIndex]
    local originalMediaItem = MediaItems[mediaItem.originalIndex]

    if (originalMediaItem:isImage()) then
        mediaItemDuration = suggestedDuration
    else
        mediaItemDuration = math.min(originalMediaItem:getDurationSec(), suggestedDuration)
    end

    -- loop through events
    for _, mediaEvent in ipairs(MediaEvents) do
        if mediaEvent:getSourceMedia() == "config" then goto CONTINUE end

        local numEvents = mediaEvent:size()
        local prevTransitionTime = 0
        local didAddClip = false
        local leftBound = mediaItemDuration - _trimThreshold     -- time stamp for left bound of selected media item
        local rightBound = mediaItemDuration + _trimThreshold    -- time stamp for right bound of selected media item
        local candidates = {}   -- table for transition candidates. candidate example: { onset = true, weights = 1.2, diffToSuggestion: 0.24 }

        -- if no media event exists, make transitions at suggested duration
        if (numEvents == 0) then
            generateProjectWithEmptyBeatEvents(suggestedDuration)
            return
        end

        -- loop through media events
        for i = 0, numEvents - 1, 1 do
            -- check 'beat' tag in an event and make transition
            local startTime, weight, duration, tags = mediaEvent:getEvent(i)
            if (table.contains(tags, "beat")) then
                if (startTime >= leftBound and startTime <= rightBound) then
                    local candidate = {
                        start = startTime,
                        onset = table.contains(tags, "onset"),
                        weights = weight,
                        diffToSuggestion = math.abs(prevTransitionTime + mediaItemDuration - startTime)
                    }
                    table.insert(candidates, candidate)
                end
            end

            -- beat passed right bound
            if (startTime > rightBound) then
                local clipDuration = mediaItemDuration
                local bestBeat = pickBestBeat(candidates)
                if bestBeat ~= nil then
                    clipDuration = bestBeat.start - prevTransitionTime
                end

                Project:addClip(mediaItem.startTime, clipDuration, 1.0, originalMediaItem)

                if (originalMediaItem:isImage()) then
                    addTreatment()
                end

                Treatments:sepiaTreatment(0.0, clipDuration, SEPIA)

                prevTransitionTime = prevTransitionTime + clipDuration

                if shouldAddFadeTransition(mediaItemIndex, mediaItemsSize) then
                    Project:addTransition("fade", _transitionDuration)
                    prevTransitionTime = prevTransitionTime - _transitionDuration
                end

                didAddClip = true
                candidates = {}
                i = i - 1
            end

            -- update params for next media item
            if (didAddClip) then
                mediaItemIndex = mediaItemIndex + 1

                if (mediaItemIndex > mediaItemsSize) then
                    break
                end

                mediaItem = _mediaItemsWithSegments[mediaItemIndex]
                originalMediaItem = MediaItems[mediaItem.originalIndex]

                if (originalMediaItem:isImage()) then
                    mediaItemDuration = suggestedDuration
                else
                    mediaItemDuration = math.min(originalMediaItem:getDurationSec() - mediaItem.startTime, suggestedDuration)
                end

                leftBound = prevTransitionTime + mediaItemDuration - _trimThreshold
                rightBound = prevTransitionTime + mediaItemDuration + _trimThreshold

                didAddClip = false
            end
        end

        -- handle remaining items
        while (mediaItemIndex <= mediaItemsSize) do
            mediaItem = _mediaItemsWithSegments[mediaItemIndex]
            originalMediaItem = MediaItems[mediaItem.originalIndex]

            if (originalMediaItem:isImage()) then
                mediaItemDuration = suggestedDuration
            else
                mediaItemDuration = math.min(originalMediaItem:getDurationSec() - mediaItem.startTime, suggestedDuration)
            end

            Project:addClip(mediaItem.startTime, mediaItemDuration, 1.0, originalMediaItem)

            if (originalMediaItem:isImage()) then
                addTreatment()
            end

            Treatments:sepiaTreatment(0.0, mediaItemDuration, SEPIA)

            if shouldAddFadeTransition(mediaItemIndex, mediaItemsSize) then
                Project:addTransition("fade", _transitionDuration)
            end

            mediaItemIndex = mediaItemIndex + 1
        end

        Project:addEffect("fadeOut", _fadeOutDuration)

        ::CONTINUE::
    end
end

main()
