-- (c) Meta Platforms, Inc. and affiliates. Confidential and proprietary.

-- Get the median of a table.
function table.median( t )
  local temp={}

  -- deep copy table so that when we sort it, the original is unchanged
  -- also weed out any non numbers
  for k,v in pairs(t) do
    if type(v) == 'number' then
      table.insert( temp, v )
    end
  end

  table.sort( temp )

  -- If we have an even number of table elements or odd.
  if math.fmod(#temp,2) == 0 then
    -- return mean value of middle two elements
    return ( temp[#temp/2] + temp[(#temp/2)+1] ) / 2
  else
    -- return middle element
    return temp[math.ceil(#temp/2)]
  end
end

function table.contains(table, element)
  for _, value in pairs(table) do
      if value == element then
          return true
      end
  end
  return false
end

function getConfig()
  -- default config
  local config = {
      maxItemCount = 15,
      isVideoSegmentationEnabled = false,
      minVideoSegmentDuration = 30,
      seedValue = 0
  }

  for _, mediaEvent in ipairs(MediaEvents) do
      if mediaEvent:getSourceMedia() == "config" then
          for i = 0, mediaEvent:size() - 1, 1 do
              local startTime, weight, duration, tags = mediaEvent:getEvent(i)
              local param = tags[1]
              if param == "maxItemCount" then
                  config.maxItemCount = math.floor(weight)
              elseif param == "isVideoSegmentationEnabled" then
                  config.isVideoSegmentationEnabled = weight == 1.0
              elseif param == "minVideoSegmentDuration" then
                  config.minVideoSegmentDuration = math.floor(weight)
              elseif param == "seed" then
                  config.seedValue = math.floor(weight)
              end
          end
      end
  end

  return config
end

function generateMediaItemsWithSegments(config)
  local mediaItemsCount = #MediaItems
  local availableSegmentCount = config.maxItemCount - mediaItemsCount
  local list = {}
  local offset = 0
  for i = 1, mediaItemsCount, 1 do
      local item = MediaItems[i]
      local startTime = item:getStartSec()
      local duration = item:getDurationSec()

      list[i + offset] = {}
      list[i + offset]["originalIndex"] = i
      list[i + offset]["startTime"] =  startTime

      if config.isVideoSegmentationEnabled and not item:isImage() and duration > config.minVideoSegmentDuration and availableSegmentCount > 0 then
          local interval = duration / 3
          for j = 1, 2, 1 do
              if availableSegmentCount > 0 then
                  offset = offset + 1
                  availableSegmentCount = availableSegmentCount - 1

                  list[i + offset] = {}
                  list[i + offset]["originalIndex"] = i
                  list[i + offset]["startTime"] = startTime + (interval * j)
              end
          end
      end
  end
  return list
end

function isMediaEventsEmpty()
  local configFilesAmt = 0
  for _, mediaEvent in ipairs(MediaEvents) do
      if mediaEvent:getSourceMedia() == "config" then
          configFilesAmt = configFilesAmt + 1
      end
  end

  local resultingTableSize = table.size(MediaEvents) - configFilesAmt
  return resultingTableSize == 0

end

function downsampleToHalf(inputBeats, preferUpbeats)
  local evenArray = {}
  local evenWeightAvg = 0
  local oddArray = {}
  local oddWeightAvg = 0

  for i, inputBeat in ipairs(inputBeats) do
      if i % 2 == 0 then
          table.insert(oddArray, inputBeat)
          oddWeightAvg = oddWeightAvg + inputBeat.weight
          print("i: " .. i .. " BEAT START: " .. inputBeat.startTime)
      else
          table.insert(evenArray, inputBeat)
          evenWeightAvg = evenWeightAvg + inputBeat.weight
          print("i: " .. i .. " BEAT START: " .. inputBeat.startTime)
      end
  end
  oddWeightAvg = oddWeightAvg / #oddArray
  evenWeightAvg = evenWeightAvg / #evenArray
  if oddWeightAvg < evenWeightAvg then
    if preferUpbeats then
      return oddArray
    end
  end
  return evenArray
end

-- Add more beats to double the rhythm
function doubleRhythm(beatsTable, beatDuration)
  local doubleBeatsTable = {}
  for i, beatEvent in ipairs(beatsTable) do
    local newBeat = {start = beatEvent.start + beatDuration / 2}
    table.insert(doubleBeatsTable, beatEvent)
    table.insert(doubleBeatsTable, newBeat)
  end
  return doubleBeatsTable, beatDuration / 2
end

function mediaEventToTable(mediaEvent, timeAdjustment)
  local numEvents = mediaEvent:size()
  local eventsTable = {}
  for i = 0, numEvents - 1, 1 do
      local start, weights, _, tag = mediaEvent:getEvent(i)
      local event = {
          startTime = start - timeAdjustment, -- Adjust cut time
          weight = weights,
          tags = tag
      }
      table.insert(eventsTable, event)
  end
  return eventsTable
end

function findBPM(events)
  local distances = {}
  local lastBeat = 0.0
  for _, event in ipairs(events) do
      if (table.contains(event.tags, "beat")) then
          table.insert(distances, event.startTime - lastBeat)
          lastBeat = event.startTime
      end
  end
  return math.floor(60 / table.median(distances))
end

-- Extract only the beats and other info about them
function getBeatInfo()
  local beatsTable = {}
  local prevStart = 0
  local beatDurations = {}
  local medianDuration = -1
  for _, mediaEvent in ipairs(MediaEvents) do
    if mediaEvent:getSourceMedia() == "config" then goto CONTINUE end
      local numEvents = mediaEvent:size()
      local onsetsAfter = 0
      for i = 0, numEvents - 1, 1 do
          -- check 'beat' tag in an event
          local start, weights, _, tags = mediaEvent:getEvent(i)
          if (table.contains(tags, "beat")) then
              -- save previous beat's onsets after the beat
              if #beatsTable ~= 0 then
                  beatsTable[#beatsTable].onsetsAfter = onsetsAfter
              end
              -- save the new beat
              local beat = {
                  startTime = start,
                  weight = weights,
                  onset = table.contains(tags, "onset"),
                  onsetsBefore = onsetsAfter, -- onsets before beat are previous beat's onsets after the beat
                  onsetsAfter = 0, -- This will be updated by the next beat.
              }
              table.insert(beatsTable, beat)
              onsetsAfter = 0
              -- save the beat duration
              local currDuration = start - prevStart
              table.insert(beatDurations, currDuration)
              prevStart = start
          else
              if #beatsTable ~= 0 then
                  onsetsAfter = onsetsAfter + 1
              end
          end
      end
    ::CONTINUE::
  end
  if #beatDurations > 1 then
    medianDuration = table.median(beatDurations)
  end
  return beatsTable, medianDuration
end
